{*****************************************************************************************}
{* Projekt : S5Link    Komponente zur Kopplung S5 <--> PC fr Delphi unter Win 95/NT     *}
{*****************************************************************************************}

unit S5Link;

{*** Interfaceteil der Unit ***************************************************}

interface

{*** eingebundene Units *******************************************************}

uses
  Windows,
  Messages,
  SysUtils,
  Classes,
  Graphics,
  Controls,
  Forms,
  StdCtrls,
  Dialogs, ExtCtrls;

{*** Feste ResourceStrings ****************************************************}

resourcestring
  csNoError          = 'Keine Fehler';
  csWrongMode        = 'Falsche Betriebsart';
  csUnknownCmd       = 'Funktion im AG unbekannt';
  csNoMem            = 'Speicherplatz nicht ausreichend';
  csBstExist         = 'Baustein bereits vorhanden';
  csBstExistInEPROM  = 'Baustein bereits im EPROM vorhanden';
  csBstListLocked    = 'Bausteinliste ist gesperrt';
  csNoBreakPoint     = 'Haltepunkt nicht im Programm';
  csBstNotExist      = 'Baustein ist nicht vorhanden';
  csNoInit           = 'Fehler beim initialisieren';
  csNoPort           = 'COM - Port konnte nicht geffnet werden';
  csDontKnow         = 'Unbekannter Fehler';

{*** Konstantendefinitionen ***************************************************}

const
  NUL                          = $00;
  SOH                          = $01;
  STX                          = $02;
  ETX                          = $03;
  EOT                          = $04;
  ENQ                          = $05;
  ACK                          = $06;
  BEL                          = $07;
  BS                           = $08;
  HT                           = $09;
  LF                           = $0A;
  VT                           = $0B;
  FF                           = $0C;
  CR                           = $0D;
  SO                           = $0E;
  SI                           = $0F;
  DLE                          = $10;
  DC1                          = $11;
  DC2                          = $12;
  DC3                          = $13;
  DC4                          = $14;
  NAK                          = $15;
  SYN                          = $16;
  ETB                          = $17;
  CAN                          = $18;
  EM                           = $19;
  SUB                          = $1A;
  ESC                          = $1B;
  FS                           = $1C;
  GS                           = $1D;
  RS                           = $1E;
  US                           = $1F;

  BST_ART_ANZ                  = 7;
  DB                           = $01;
  SB                           = $02;
  PB                           = $03;
  FB                           = $04;
  OB                           = $05;
  FX                           = $06;
  DX                           = $07;

  E                            = $10;
  A                            = $11;
  M                            = $12;
  T                            = $13;
  Z                            = $14;

  BausteinType                 : Array[1..BST_ART_ANZ] of Byte =
                                       ($01, $02, $04, $08, $10, $05, $0C);
  BausteinName                 : Array[1..BST_ART_ANZ] of String[2] =
                                       ('DB', 'SB', 'PB', 'FB', 'OB', 'FX', 'DX');
  TimerStr                     : Array[0..3] of String[4] =
                                       ('0.01', '0.10', '1.0', '10');
  wTimeOut                     = 5000;


  S5_E_NOERR                   = $0000;
  S5_E_WRONGMODE_A             = $0002;
  S5_E_UNKNOWNCMD              = $0003;
  S5_E_NOMEM                   = $0004;
  S5_E_BSTEXIST                = $0005;
  S5_E_BSTEXISTINEPROM         = $0006;
  S5_E_BSTLISTLOCKED           = $0007;
  S5_E_NOBREAKPOINT            = $0008;
  S5_E_WRONGMODE_B             = $000D;
  S5_E_BSTNOTEXIST             = $0014;
  S5_E_NOINIT                  = $FFFE;
  S5_E_NOPORT                  = $FFFF;

{*** Datentypendefinitionen ***************************************************}

type
  TInfo                        = record
    AdrPE                      : Word;  // Adresse Peripherie der Eingnge
    AdrPA                      : Word;  // Adresse Peripherie der Ausgnge
    AdrPAE                     : Word;  // Adresse Prozessabbild der Eingnge
    AdrPAA                     : Word;  // Adresse Prozessabbild der Ausgnge
    AdrM                       : Word;  // Adresse Prozessabbild der Merker
    AdrT                       : Word;  // Adresse Prozessabbild der Timer
    AdrZ                       : Word;  // Adresse Prozessabbild der Zhler
    AdrSD                      : Word;  // Adresse Prozessabbild der Systemdaten
    AGSoftwarestand            : Word;
    AdrMemEnd                  : Word;
    AdrSysMem                  : Word;  // Adresse des Systemspeichers
    LaengeDerBstListe          : Array[1..BST_ART_ANZ] of Word;
    LaengeDesDB0               : Word;
    CPUKennung                 : Word;
    BstKopfLaenge              : Word; // Bausteinkopflnge
    ZweiteCPUKennung           : Word;
  end;

  TBaustein                    = record
    SynchronMuster             : Word;
    Kennung                    : Word;
    PGKennung                  : Byte;
    BibNr                      : Array[1..3] of Byte;
    LaengeMitKopf              : Word;
    Daten                      : Array[0..8192] of Byte;
  end;

  TBstInfo                     = record
    BausteinAdresse            : Word;
    SynchronMuster             : Word;
    Kennung                    : Word;
    PGKennung                  : Byte;
    BibNr                      : Array[1..3] of Byte;
    LaengeMitKopf              : Word;
  end;

  TPLC                         = record
    Com                        : Word;
    Initialisiert              : Boolean;
    System                     : TInfo;
    Error                      : Word;
  end;

  TByteBuffer                  = Array[0..16384] of Byte;
  PByteBuffer                  = ^TByteBuffer;
  TWordBuffer                  = Array[0..8192] of Word;
  PWordBuffer                  = ^TByteBuffer;
  TResultBuffer                = Array[0..255] of Byte;
  PResultBuffer                = ^TByteBuffer;

  TPort       = (COM1, COM2, COM3, COM4);
  TBstTyp     = (tDB, tSB, tPB, tFB, tOB, tFX, tDX);
  TCommand    = (ReadEB, ReadEW, ReadAB, ReadAW, ReadMB, ReadMW, ReadDBData, ReadBst,
                 ReadTimer, ReadCounter, ReadMem,
                 WriteAB, WriteAW, WriteMB, WriteMW, WriteDBData, WriteBst, WriteMem,
                 BstDir, Start, Stop, Compress, DelBst, CreateBst);

{*** Objectdefinition **********************************************************}

  TS5Link = class(TComponent)
  private
    FActive      : Boolean;
    FPort        : TPort;
    FCommand     : TCommand;
    FBstNr       : Byte;
    FAbWort      : Byte;
    FAnzahl      : Byte;
    FBstTyp      : TBstTyp;
    FAnfAdr      : Word;
    FEndAdr      : Word;
  protected
    FCID         : THandle ;
    FDCB         : TDCB ;
    FCTO         : TCommTimeOuts;
    OverlapBlock : TOverlapped;
    ByteBuffer   : TResultBuffer;
    abBuffer     : TByteBuffer;
    awBuffer     : TWordBuffer;
    dwDummy      : DWord;
    Baustein     : TBaustein;
    procedure   SetPort(Port : TPort);
    procedure   SetCommand(Command : TCommand);
    procedure   SetActive(boActive : Boolean);
    procedure   SetBstNr(BstNr : Byte);
    procedure   SetBstTyp(BstTyp : TBstTyp);
    procedure   SetAbWort(AbWort : Byte);
    procedure   SetAnzahl(Anzahl : Byte);
    procedure   SetAnfAdr(AnfAdr : Word);
    procedure   SetEndAdr(EndAdr : Word);
    function    OpenCOM(No: Byte) : Boolean;
    function    CloseCOM : Boolean;
    function    ReadData(var DataBuffer; Anzahl : DWord) : Boolean;
    function    WriteData(var DataBuffer; Anzahl : DWord) : Boolean;
    function    FunktionEinleiten(var PLC: TPLC; FNr : Byte) : Boolean;
    function    FunktionFortfuehren(var PLC: TPLC) : Boolean;
    function    AbschlussMeldung(var PLC : TPLC) : Boolean;
    function    FunktionAbschliessen(var PLC : TPLC) : Boolean;
    function    AgSystemParameter(var PLC: TPLC) : Boolean;
    function    AgInit(var PLC: TPLC) : Boolean;
    function    AgExit(var PLC: TPLC) : Boolean;
    function    AgBausteinDir(var PLC: TPLC; BstTyp: Byte; Buffer: Pointer) : Boolean;
    function    AgBausteinAusgabe(var PLC: TPLC; BstTyp, BstNr: Byte; Buffer: Pointer) : Boolean;
    function    AgBausteinEingabe(var PLC: TPLC; BstTyp, BstNr: Byte; Anzahl : Word; Buffer: Pointer) : Boolean;
    function    AgBausteinAusgabeEx(var PLC: TPLC; BstTyp, BstNr: Byte; AbWort, Anzahl : Word; Buffer: Pointer) : Boolean;
    function    AgRead(var PLC: TPLC; AnfangsAdr, EndAdr: Word; Buffer : Pointer) : Boolean;
    function    AgReadTimer(var PLC : TPLC; TimNr : Byte; Buffer : Pointer) : Boolean;
    function    AgReadCounter(var PLC : TPLC; CntNr : Byte; Buffer : Pointer) : Boolean;
    function    AgReadEB(var PLC: TPLC; ByteNr: Byte; Buffer: Pointer) : Boolean;
    function    AgReadEW(var PLC: TPLC; WortNr: Byte; Buffer: Pointer) : Boolean;
    function    AgReadAB(var PLC: TPLC; ByteNr: Byte; Buffer: Pointer) : Boolean;
    function    AgReadAW(var PLC: TPLC; WortNr: Byte; Buffer: Pointer) : Boolean;
    function    AgReadMB(var PLC: TPLC; ByteNr: Byte; Buffer: Pointer) : Boolean;
    function    AgReadMW(var PLC: TPLC; WortNr: Byte; Buffer: Pointer) : Boolean;
    function    AgReadDataBlock(var PLC : TPLC; BstNr : Byte; AbWort, Anzahl : Word; Buffer : Pointer) : Boolean;
    function    AgWriteDataBlock(var PLC : TPLC; BstNr : Byte; AbWort, Anzahl : Word; Buffer : Pointer) : Boolean;
    function    AgWriteAB(var PLC: TPLC; ByteNr: Byte; Buffer: Pointer) : Boolean;
    function    AgWriteAW(var PLC: TPLC; WortNr: Byte; Buffer: Pointer) : Boolean;
    function    AgWriteMB(var PLC: TPLC; ByteNr: Byte; Buffer: Pointer) : Boolean;
    function    AgWriteMW(var PLC: TPLC; WortNr: Byte; Buffer: Pointer) : Boolean;
    function    AgWrite(var PLC: TPLC; AnfangsAdr, EndAdr: Word; Buffer : Pointer) : Boolean;
    function    AgBausteinInfo(var PLC: TPLC; BstTyp, BstNr: Byte; Buffer: Pointer) : Boolean;
    function    AgReadData(var PLC : TPLC; Typ , BstNr, AbByte, Anzahl : Byte; Buffer : Pointer) : Boolean;
    function    AgWriteData(var PLC : TPLC; Typ , BstNr : Byte; AbByte, Anzahl : Word; Buffer : Pointer) : Boolean;
    function    AgStart(var PLC: TPLC) : Boolean;
    function    AgStop(var PLC: TPLC) : Boolean;
    function    AgCompress(var PLC: TPLC) : Boolean;
    function    AgDelBst(var PLC: TPLC; BstTyp, BstNr : Byte) : Boolean;
    function    AgCreateBst(var PLC : TPLC; BstTyp, BstNr : Byte; Anzahl : Word) : Boolean;
  public
    PLC         : TPLC;
    constructor Create(aOwner: TComponent);                  override;
    destructor  Destroy;                                     override;
    function    CalcAdr(PLC: TPLC; Typ, BstNr, AbByte, Anzahl: Byte; var wAnfAdr, wEndAdr : Word) : Boolean;
    function    CheckPLC(var PLC : TPLC) : Boolean;
    function    CheckBatt(var PLC : TPLC) : Boolean;
    function    GetErrText(wErrNum : Word) : String;
    function    GetPLCTyp(var PLC : TPLC) : String;
    function    AgDecodeBst(var PLC : TPLC; Bst : TBaustein; boAdressen : Boolean) : TStringList;
  published
    function    Execute(pBuffer : Pointer) : Boolean;
    property    Active  : Boolean read FActive write SetActive default False;
    property    Port    : TPort read FPort write SetPort default COM1;
    property    Command : TCommand read FCommand write SetCommand default ReadEB;
    property    BstNr   : Byte read FBstNr write SetBstNr default 1;
    property    BstTyp  : TBstTyp read FBstTyp write SetBstTyp default tDB;
    property    AbWort  : Byte read FABWort write SetAbwort default 0;
    property    Anzahl  : Byte read FAnzahl write SetAnzahl default 1;
    property    AnfAdr  : Word read FAnfAdr write SetAnfAdr default 0;
    property    EndAdr  : Word read FEndAdr write SetEndAdr default 1;
  end;

  TInfoForm = class(TForm)
    Label1: TLabel;
    Image1: TImage;
    Bevel1: TBevel;
  private
    { Private-Deklarationen }
  public
    { Public-Deklarationen }
  end;

{*** Allgemeine Variablen *****************************************************}

var
  InfoForm : TInfoForm;

{--- Hilfsfunktionen und Utils ------------------------------------------------}

function    Exponent(Base, Expo: Integer) : LongInt;
function    HexCharToWert(HexChar: Char) : Word;
function    HexToInt(HexWert: String): LongInt;
function    BinToInt(BinWert: String) : LongInt;
function    IntToBin(IntWert: LongInt) : String;
function    WordToBin(wData : Word) : String;
function    ByteToBin(bData : Byte) : String;
function    ByteToBCD(bDez : Byte) : Byte;
function    WordToBCD(wDez : Word) : Word;
function    SureDlg(sText : String) : Boolean;
function    TestBit(wIn : Word; bBit : Byte) : Boolean;
function    SetBit(wIn : Word; bBit : Byte) : Word;
function    ClearBit(wIn : Word; bBit : Byte) : Word;
function    ChangeBit(wIn : Word; bBit : Byte) : Word;
procedure   DecodeTimer(wTimer : Word; var TimerVal, TimerRes : Word);
procedure   DecodeCounter(wCounter : Word; var CounterVal : Word);
procedure   InfoDlg(sText : String);
procedure   ShowInfo(sCaption, sInfo : String);
procedure   HideInfo;
procedure   Register;


{*** Implementationsteil *******************************************************}

implementation

{*** eingebundene Formulare ***************************************************}

{$R *.DFM}

{------------------------------------------------------------------------------}

constructor TS5Link.Create(aOwner: TComponent);
begin
  inherited;
  FPort := COM1;
  FBstTyp := tDB;
  FBstNr := 1;
  FAbwort := 0;
  FAnzahl := 1;
  FCommand := ReadEB;
end;

{------------------------------------------------------------------------------}

destructor  TS5Link.Destroy;
begin

  inherited;
end;

{------------------------------------------------------------------------------}

procedure TS5Link.SetActive(boActive : Boolean);
begin
  if boActive then begin
    PLC.Com := Succ(Byte(FPort));
    FActive := AgInit(PLC);
  end else begin
    AgExit(PLC);
    FActive := False;
  end;
end;

{------------------------------------------------------------------------------}

procedure TS5Link.SetPort(Port : TPort);
begin
  FPort := Port;
  PLC.Com := Byte(Port) + 1;
end;

{------------------------------------------------------------------------------}

procedure TS5Link.SetCommand(Command : TCommand);
begin
  FCommand := Command;
end;

{------------------------------------------------------------------------------}

procedure TS5Link.SetBstNr(BstNr : Byte);
begin
  FBstNr := BstNr;
end;

{------------------------------------------------------------------------------}

procedure TS5Link.SetBstTyp(BstTyp : TBstTyp);
begin
  FBstTyp := BstTyp;
end;

{------------------------------------------------------------------------------}

procedure TS5Link.SetAbWort(AbWort : Byte);
begin
  FAbWort := AbWort;
end;

{------------------------------------------------------------------------------}

procedure TS5Link.SetAnzahl(Anzahl : Byte);
begin
  FAnzahl := Anzahl;
end;

{------------------------------------------------------------------------------}

procedure TS5Link.SetAnfAdr(AnfAdr : Word);
begin
  FAnfAdr := AnfAdr;
end;

{------------------------------------------------------------------------------}

procedure TS5Link.SetEndAdr(EndAdr : Word);
begin
  FEndAdr := EndAdr;
end;

{------------------------------------------------------------------------------}

function TS5Link.Execute(pBuffer : Pointer) : Boolean;
begin
  PLC.Error := 0;
  case FCommand of
    ReadEB      : Result := AgReadEB(PLC, FBstNr, pBuffer);
    ReadEW      : Result := AgReadEW(PLC, FBstNr, pBuffer);
    ReadAB      : Result := AgReadAB(PLC, FBstNr, pBuffer);
    ReadAW      : Result := AgReadAW(PLC, FBstNr, pBuffer);
    ReadMB      : Result := AgReadMB(PLC, FBstNr, pBuffer);
    ReadMW      : Result := AgReadMW(PLC, FBstNr, pBuffer);
    ReadDBData  : Result := AgReadDataBlock(PLC, FBstNr, FAbwort, FAnzahl, pBuffer);
    ReadBst     : Result := AgBausteinAusgabe(PLC, Byte(FBstTyp) + 1, FBstNr, pBuffer);
    ReadTimer   : Result := AgReadTimer(PLC, BstNr, pBuffer);
    ReadCounter : Result := AgReadCounter(PLC, BstNr, pBuffer);
    ReadMem     : Result := AgRead(PLC, FAnfAdr, FEndAdr, pBuffer);
    WriteAB     : Result := AgWriteAB(PLC, FBstNr, pBuffer);
    WriteAW     : Result := AgWriteAW(PLC, FBstNr, pBuffer);
    WriteMB     : Result := AgWriteMB(PLC, FBstNr, pBuffer);
    WriteMW     : Result := AgWriteMW(PLC, FBstNr, pBuffer);
    WriteDBData : Result := AgWriteDataBlock(PLC, FBstNr, FAbwort, FAnzahl, pBuffer);
    WriteBst    : Result := AgBausteinEingabe(PLC, Byte(FBstTyp) + 1, FBstNr, FAnzahl, pBuffer);
    BstDir      : Result := AgBausteinDir(PLC, Byte(FBstTyp) + 1, pBuffer);
    Start       : Result := AgStart(PLC);
    Stop        : Result := AgStop(PLC);
    Compress    : Result := AgCompress(PLC);
    DelBst      : Result := AgDelBst(PLC, Byte(FBstTyp) + 1, FBstNr);
    CreateBst   : Result := AgCreateBst(PLC, Byte(FBstTyp) + 1, FBstNr, FAnzahl);
    WriteMem    : Result := AgRead(PLC, FAnfAdr, FEndAdr, pBuffer);
    else          Result := False;
  end;
end;

{------------------------------------------------------------------------------}

function TS5Link.OpenCOM(No: Byte) : Boolean;
var acDummy  : Array[0..4] of Char;
begin
  OverlapBlock.Offset := 0;
  OverlapBlock.OffsetHigh := 0;
  OverlapBlock.Internal := 0;
  OverlapBlock.InternalHigh := 0;
  OverlapBlock.hEvent :=  CreateEvent(NIL, FALSE, TRUE, pChar('S5_Event')) ;
  StrPCopy(acDummy, 'COM' + IntToStr(No));
  FCID := CreateFile(acDummy, (GENERIC_READ or GENERIC_WRITE), 0, NIL, OPEN_EXISTING, FILE_FLAG_OVERLAPPED, 0);
  if FCID <> INVALID_HANDLE_VALUE then begin
    Result := True;
    SetupComm (FCID, 1024, 1024);
    FDCB.DCBLength := SizeOf(TDCB);
    FDCB.BaudRate := CBR_9600;
    FDCB.ByteSize := 8;
    FDCB.Parity := EVENPARITY;
    FDCB.StopBits := ONESTOPBIT;
    FDCB.Flags := $0001;
    SetCommState(FCID, FDCB);
    FCTO.ReadIntervalTimeout := 0;
    FCTO.ReadTotalTimeoutMultiplier := 0;
    FCTO.ReadTotalTimeoutConstant := 5000;
    FCTO.WriteTotalTimeoutMultiplier := 0;
    FCTO.WriteTotalTimeoutConstant := 5000;
    SetCommTimeouts(FCID, FCTO);
    EscapeCommFunction(FCID, SETRTS);
    EscapeCommFunction(FCID, SETDTR);
  end else
    Result := False;
end;

{------------------------------------------------------------------------------}

function TS5Link.CloseCOM : Boolean;
begin
  EscapeCommFunction(FCID, CLRRTS);
  EscapeCommFunction(FCID, CLRDTR);
  SetCommMask(FCID, 0);
  Result := CloseHandle(FCID);
end;

{------------------------------------------------------------------------------}

function TS5Link.ReadData(var DataBuffer; Anzahl : DWord) : Boolean;
var Res  : Dword;
begin
  Result := False;
  if not ReadFile(FCID, DataBuffer, Anzahl, dwDummy, @OverlapBlock) then begin
    Res := WaitForSingleObject(OverlapBlock.hEvent, wTimeOut);
    case Res of
      WAIT_OBJECT_0 : begin
                        ResetEvent(OverlapBlock.hEvent);
                        OverlapBlock.Offset := 0;
                        OverlapBlock.OffsetHigh := 0;
                        Result := True;
                      end;
      WAIT_TIMEOUT :  begin
                        ResetEvent(OverlapBlock.hEvent);
                        OverlapBlock.Offset := 0;
                        OverlapBlock.OffsetHigh := 0;
                        Result := False;
                      end;
    end;
  end;
end;

{------------------------------------------------------------------------------}

function TS5Link.WriteData(var DataBuffer; Anzahl : DWord) : Boolean;
var Res : DWord;
begin
  Result := True;
  if not WriteFile(FCID, DataBuffer, Anzahl, dwDummy, @OverlapBlock) then begin
    Result := False;
    Res := WaitForSingleObject(OverlapBlock.hEvent, wTimeOut);
    if  Res <> 0 then
      Exit
    else begin
      ResetEvent(OverlapBlock.hEvent);
      OverlapBlock.Offset := 0;
      OverlapBlock.OffsetHigh := 0;
    end;
  end;
end;

{------------------------------------------------------------------------------}

function TS5Link.FunktionEinleiten(var PLC: TPLC; FNr : Byte) : Boolean;
begin
  Result := False;
  ByteBuffer[0] := STX;
  WriteData(ByteBuffer, 1);
  ReadData(ByteBuffer, 2);
  if (ByteBuffer[0] = DLE) and (ByteBuffer[1] = ACK) then begin
    ByteBuffer[0] := FNr;
    WriteData(ByteBuffer, 1);
    ReadData(ByteBuffer, 1);
    if (ByteBuffer[0] = STX) then begin
      ByteBuffer[0] := DLE;
      ByteBuffer[1] := ACK;
      WriteData(ByteBuffer, 2);
      ReadData(ByteBuffer, 3);
      if (ByteBuffer[0] = SYN) and (ByteBuffer[1] = DLE) and (ByteBuffer[2] = ETX) then begin
        ByteBuffer[0] := DLE;
        ByteBuffer[1] := ACK;
        WriteData(ByteBuffer, 2);
        Result := True;
      end else begin
        PLC.Error := (ByteBuffer[0]);
        FunktionAbschliessen(PLC);
      end;
   end;
  end;
end;

{------------------------------------------------------------------------------}

function TS5Link.FunktionFortfuehren(var PLC: TPLC) : Boolean;
begin
  Result := False;
  ReadData(ByteBuffer, 3);
  if (ByteBuffer[0] = DLE) and (ByteBuffer[1] = ACK) and (ByteBuffer[2] = STX) then begin
    ByteBuffer[0] := DLE;
    ByteBuffer[1] := ACK;
    WriteData(ByteBuffer, 2);
    ReadData(ByteBuffer, 1);
    if (ByteBuffer[0] = NUL) then begin
      Result := True;
    end else begin
      PLC.Error := (ByteBuffer[0]);
      FunktionAbschliessen(PLC);
    end;
  end;
end;

{------------------------------------------------------------------------------}

function TS5Link.AbschlussMeldung(var PLC : TPLC) : Boolean;
begin
  Result := False;
  ReadData(ByteBuffer, 1);
  if (ByteBuffer[0] = STX) then begin
    ByteBuffer[0] := DLE;
    ByteBuffer[1] := ACK;
    WriteData(ByteBuffer, 2);
    ReadData(ByteBuffer, 3);
    if (ByteBuffer[0] = DC2) and (ByteBuffer[1] = DLE) and (ByteBuffer[2] = ETX) then begin
      ByteBuffer[0] := DLE;
      ByteBuffer[1] := ACK;
      WriteData(ByteBuffer, 2);
      Result := True;
    end;
  end;
end;

{------------------------------------------------------------------------------}

function TS5Link.FunktionAbschliessen(var PLC : TPLC) : Boolean;
begin
  Result := False;
  ReadData(ByteBuffer, 2);
  if (ByteBuffer[0] = DLE) and (ByteBuffer[1] = ETX) then begin
    ByteBuffer[0] := DLE;
    ByteBuffer[1] := ACK;
    WriteData(ByteBuffer, 2);
    Result := AbschlussMeldung(PLC);
  end;
end;

{------------------------------------------------------------------------------}

function TS5Link.AgSystemParameter(var PLC: TPLC) : Boolean;
begin
  Result := False;
  if FunktionEinleiten(PLC, $18) then begin
    ByteBuffer[0] := DLE;
    ByteBuffer[1] := EOT;
    WriteData(ByteBuffer, 2);
    if FunktionFortfuehren(PLC) then
      ReadData(PLC.System, SizeOf(TInfo));
    Result := FunktionAbschliessen(PLC);
  end;
  if Result then begin
    PLC.System.AdrPE := Swap(PLC.System.AdrPE);
    PLC.System.AdrPA := Swap(PLC.System.AdrPA);
    PLC.System.AdrPAE := Swap(PLC.System.AdrPAE);
    PLC.System.AdrPAA := Swap(PLC.System.AdrPAA);
    PLC.System.AdrM := Swap(PLC.System.AdrM);
    PLC.System.AdrT := Swap(PLC.System.AdrT);
    PLC.System.AdrZ := Swap(PLC.System.AdrZ);
    PLC.System.AdrSD := Swap(PLC.System.AdrSD);
    PLC.System.AGSoftwarestand := Swap(PLC.System.AGSoftwarestand);
    PLC.System.AdrMemEnd := Swap(PLC.System.AdrMemEnd);
    PLC.System.AdrSysMem := Swap(PLC.System.AdrSysMem);
    PLC.System.LaengeDerBstListe[DB] := Swap(PLC.System.LaengeDerBstListe[DB]);
    PLC.System.LaengeDerBstListe[SB] := Swap(PLC.System.LaengeDerBstListe[SB]);
    PLC.System.LaengeDerBstListe[PB] := Swap(PLC.System.LaengeDerBstListe[PB]);
    PLC.System.LaengeDerBstListe[FB] := Swap(PLC.System.LaengeDerBstListe[FB]);
    PLC.System.LaengeDerBstListe[OB] := Swap(PLC.System.LaengeDerBstListe[OB]);
    PLC.System.LaengeDerBstListe[FX] := Swap(PLC.System.LaengeDerBstListe[FX]);
    PLC.System.LaengeDerBstListe[DX] := Swap(PLC.System.LaengeDerBstListe[DX]);
    PLC.System.LaengeDesDB0 := Swap(PLC.System.LaengeDesDB0);
    PLC.System.ZweiteCPUKennung := Swap(PLC.System.ZweiteCPUKennung);
    PLC.System.BstKopfLaenge := Swap(PLC.System.BstKopfLaenge);
    PLC.System.CPUKennung := Swap(PLC.System.CPUKennung);
  end;
end;

{------------------------------------------------------------------------------}

function TS5Link.AgInit(var PLC: TPLC) : Boolean;
begin
  Result := False;
  if OpenCOM(PLC.Com) then begin
    if AgSystemParameter(PLC) then begin
      Result := True;
      PLC.Error := S5_E_NOERR;
    end else begin
      CloseCom;
      PLC.Error := S5_E_NOINIT;
    end;
  end else
    PLC.Error := S5_E_NOPORT;
  PLC.Initialisiert := Result;
end;

{------------------------------------------------------------------------------}

function TS5Link.AgExit(var PLC: TPLC) : Boolean;
begin
  Result := CloseCOM;
  PLC.Initialisiert := False;
end;

{------------------------------------------------------------------------------}

function TS5Link.AgBausteinAusgabe(var PLC: TPLC; BstTyp, BstNr: Byte; Buffer : Pointer) : Boolean;
var wDummy   : Word;

begin
  Result := False;
  if FunktionEinleiten(PLC, $06) then begin
    ByteBuffer[0] := BausteinType[BstTyp];
    wDummy := 1;
    if BstTyp = OB then begin
      ByteBuffer[wDummy] := DLE;
      Inc(wDummy);
    end;
    ByteBuffer[wDummy] := BstNr;
    Inc(wDummy);
    if BstNr = $10 then begin
      ByteBuffer[wDummy] := DLE;
      Inc(wDummy);
    end;
    ByteBuffer[wDummy] := DLE;
    Inc(wDummy);
    ByteBuffer[wDummy] := ETX;
    Inc(wDummy);
    WriteData(ByteBuffer, wDummy);
    if FunktionFortfuehren(PLC) then begin
      ReadData(Baustein, PLC.System.BstKopflaenge);
      Baustein.LaengeMitKopf := Swap(Baustein.LaengeMitKopf);
      Baustein.SynchronMuster := Swap(Baustein.SynchronMuster);
      for wDummy := 0 to ((Baustein.LaengeMitKopf SHL 1) - 1 - PLC.System.BstKopflaenge) do begin
        ReadData(Baustein.Daten[wDummy], 1);
        if Baustein.Daten[wDummy] = DLE then
          ReadData(Baustein.Daten[wDummy], 1);
      end;
      if FunktionAbschliessen(PLC) then
        Result := True;
      Move(Baustein, awBuffer, Baustein.LaengeMitKopf SHL 1);
      for wDummy := 0 to Baustein.LaengeMitKopf do
        awBuffer[wDummy] := Swap(awBuffer[wDummy]);
      Move(awBuffer, Buffer^, Baustein.LaengeMitKopf SHL 1);
    end;
  end;
end;

{------------------------------------------------------------------------------}

function TS5Link.AgRead(var PLC: TPLC; AnfangsAdr, EndAdr: Word; Buffer : Pointer) : Boolean;
var Anzahl   : Word;
    bDummy   : Byte;
begin
  Result := False;
  Anzahl := (EndAdr - AnfangsAdr) + 1;
  if Anzahl > 0 then begin
    if FunktionEinleiten(PLC, $04) then begin
      bDummy := 0;
      ByteBuffer[bDummy] := Hi(AnfangsAdr);
      Inc(bDummy);
      if ByteBuffer[bDummy - 1] = $10 then begin
        ByteBuffer[bDummy] := DLE;
        Inc(bDummy);
      end;
      ByteBuffer[bDummy] := Lo(AnfangsAdr);
      Inc(bDummy);
      if ByteBuffer[bDummy - 1] = $10 then begin
        ByteBuffer[bDummy] := DLE;
        Inc(bDummy);
      end;
      ByteBuffer[bDummy] := Hi(EndAdr);
      Inc(bDummy);
      if ByteBuffer[bDummy - 1] = $10 then begin
        ByteBuffer[bDummy] := DLE;
        Inc(bDummy);
      end;
      ByteBuffer[bDummy] := Lo(EndAdr);
      Inc(bDummy);
      if ByteBuffer[bDummy - 1] = $10 then begin
        ByteBuffer[bDummy] := DLE;
        Inc(bDummy);
      end;
      ByteBuffer[bDummy] := DLE;
      Inc(bDummy);
      ByteBuffer[bDummy] := EOT;
      Inc(bDummy);
      WriteData(ByteBuffer, bDummy);
      if FunktionFortfuehren(PLC) then begin
        ReadData(ByteBuffer, 4);     //  Lckeninformation
        for bDummy := 0 to Anzahl - 1 do begin
          ReadData(abBuffer[bDummy], 1);
          if abBuffer[bDummy] = DLE then
            ReadData(abBuffer[bDummy], 1);
        end;
        Move(abBuffer, Buffer^, Anzahl);
        Result := FunktionAbschliessen(PLC);
      end;
    end;
  end;
end;

{------------------------------------------------------------------------------}

function  TS5Link.AgBausteinInfo(var PLC: TPLC; BstTyp, BstNr: Byte; Buffer: Pointer) : Boolean;
var bDummy   : Byte;
begin
  Result := False;
  if FunktionEinleiten(PLC, $1A) then begin
    ByteBuffer[0] := BausteinType[BstTyp];
    bDummy := 1;
    if BstTyp = OB then begin
      ByteBuffer[bDummy] := DLE;
      Inc(bDummy);
    end;
    ByteBuffer[bDummy] := BstNr;
    Inc(bDummy);
    if BstNr = $10 then begin
      ByteBuffer[bDummy] := DLE;
      Inc(bDummy);
    end;
    ByteBuffer[bDummy] := DLE;
    Inc(bDummy);
    ByteBuffer[bDummy] := ETX;
    Inc(bDummy);
    WriteData(ByteBuffer, bDummy);
    if FunktionFortfuehren(PLC) then begin
      for bDummy := 0 to SizeOf(TBstInfo) - 1 do begin
        ReadData(abBuffer[bDummy], 1);
        if abBuffer[bDummy] = DLE then
          ReadData(abBuffer[bDummy], 1);
      end;
      Move(abBuffer, Baustein, SizeOf(TBstInfo));
      Baustein.LaengeMitKopf := Swap(Baustein.LaengeMitKopf);
      Baustein.SynchronMuster := Swap(Baustein.SynchronMuster);
      Result := FunktionAbschliessen(PLC);
      Move(Baustein, Buffer^, SizeOf(TBstInfo));
    end;
  end;
end;

{------------------------------------------------------------------------------}

function  TS5Link.AgWrite(var PLC: TPLC; AnfangsAdr, EndAdr: Word; Buffer : Pointer) : Boolean;
var Anzahl   : Word;
    bDummy   : Byte;
begin
  Result := False;
  Anzahl := (EndAdr - AnfangsAdr) + 1;
  Move(Buffer^, abBuffer, Anzahl);
  if Anzahl > 0 then begin
    if FunktionEinleiten(PLC, $03) then begin
      ByteBuffer[0] := Hi(AnfangsAdr);
      ByteBuffer[1] := Lo(AnfangsAdr);
      WriteData(ByteBuffer, 2);
      for bDummy := 0 to Anzahl - 1 do begin
        WriteData(abBuffer[bDummy], 1);
        if abBuffer[bDummy] = $10 then
          WriteData(abBuffer[bDummy], 1);
      end;
      ByteBuffer[0] := DLE;
      ByteBuffer[1] := EOT;
      WriteData(ByteBuffer, 2);
      ReadData(ByteBuffer, 2);
      if (ByteBuffer[0] = DLE) and (ByteBuffer[1] = ACK) then
        Result := AbschlussMeldung(PLC);
    end;
  end;
end;

{------------------------------------------------------------------------------}

function TS5Link.CalcAdr(PLC: TPLC; Typ, BstNr, AbByte, Anzahl: Byte; var wAnfAdr, wEndAdr : Word) : Boolean;
var wBaseAdr : Word;
    BstInfo  : TBstInfo;
begin
  Result := False;
  case Typ of
    E  : wBaseAdr := PLC.System.AdrPAE;
    A  : wBaseAdr := PLC.System.AdrPAA;
    M  : wBaseAdr := PLC.System.AdrM;
    T  : begin
          wBaseAdr := PLC.System.AdrT;
          AbByte := AbByte SHL 1;
         end;
    Z  : begin
          wBaseAdr := PLC.System.AdrZ;
          AbByte := AbByte SHL 1;
         end;
    DB,
    PB,
    FB,
    SB,
    OB : begin
           if AgBausteinInfo(PLC, Typ, BstNr, @BstInfo) then
             wBaseAdr := BstInfo.Bausteinadresse
           else begin
             Exit;
           end;
         end;
    else
      Exit;
  end;
  if wBaseAdr = 0 then
    Exit;
  wAnfAdr := wBaseAdr + AbByte;
  wEndAdr := wAnfAdr + Anzahl - 1;
  Result := True;
end;

{------------------------------------------------------------------------------}

function TS5Link.AGReadData(var PLC : TPLC; Typ, BstNr, AbByte, Anzahl : Byte; Buffer : Pointer) : Boolean;
var wAnfAdr  : Word;
    wEndAdr  : Word;
begin
  Result := False;
  if CalcAdr(PLC, Typ, BstNr, AbByte, Anzahl, wAnfAdr, wEndAdr) then
    if AgRead(PLC, wAnfAdr, wEndAdr, Buffer) then
      Result := True;
end;

{------------------------------------------------------------------------------}

function TS5Link.AGWriteData(var PLC : TPLC; Typ , BstNr : Byte; AbByte, Anzahl : Word; Buffer : Pointer) : Boolean;
var wAnfAdr  : Word;
    wEndAdr  : Word;
begin
  Result := False;
  if CalcAdr(PLC, Typ, BstNr, AbByte, Anzahl, wAnfAdr, wEndAdr) then
    if AGWrite(PLC, wAnfAdr, wEndAdr, Buffer) then
      Result := True;
end;

{------------------------------------------------------------------------------}

function  TS5Link.AgBausteinAusgabeEx(var PLC: TPLC; BstTyp, BstNr: Byte; AbWort, Anzahl : Word; Buffer: Pointer) : Boolean;
var BstInfo   : TBstInfo;
    wAnfAdr   : Word;
    wEndAdr   : Word;
begin
  Result := False;
  if AgBausteinInfo(PLC, BstTyp, BstNr, @BstInfo) then begin
    wAnfAdr := BstInfo.BausteinAdresse + (Abwort SHL 1);
    wEndAdr := WanfAdr + (Anzahl SHL 1);
    if AgRead(PLC, wAnfAdr, wEndAdr, Buffer) then
      Result := True;
  end;
end;

{------------------------------------------------------------------------------}

function  TS5Link.AgBausteinEingabe(var PLC: TPLC; BstTyp, BstNr: Byte; Anzahl : Word; Buffer: Pointer) : Boolean;
var bDummy      : Byte;
begin
  Result := False;
  if FunktionEinleiten(PLC, $08) then begin
    ByteBuffer[0] := BausteinType[BstTyp];
    bDummy := 1;
    if BausteinType[BstTyp] = $10 then begin
      ByteBuffer[bDummy] := DLE;
      Inc(bDummy);
    end;
    ByteBuffer[bDummy] := BstNr;
    Inc(bDummy);
    if BstNr = $10 then begin
      ByteBuffer[bDummy] := DLE;
      Inc(bDummy);
    end;
    ByteBuffer[bDummy] := Hi(Anzahl SHR 1);
    Inc(bDummy);
    if ByteBuffer[bDummy] = $10 then begin
      ByteBuffer[bDummy] := DLE;
      Inc(bDummy);
    end;
    ByteBuffer[bDummy] := Lo(Anzahl SHR 1);
    Inc(bDummy);
    if ByteBuffer[bDummy] = $10 then begin
      ByteBuffer[bDummy] := DLE;
      Inc(bDummy);
    end;
    ByteBuffer[bDummy] := DLE;
    Inc(bDummy);
    ByteBuffer[bDummy] := ETX;
    Inc(bDummy);
    WriteData(ByteBuffer, bDummy);
    ReadData(ByteBuffer, 3);
    if (ByteBuffer[0] = DLE) and (ByteBuffer[1] = ACK) and (ByteBuffer[2] = STX) then begin
      ByteBuffer[0] := DLE;
      ByteBuffer[1] := ACK;
      WriteData(ByteBuffer, 2);
      ReadData(ByteBuffer, 3);
      if (ByteBuffer[0] = $09) and (ByteBuffer[1] = DLE) and (ByteBuffer[2] = ETX) then begin
        ByteBuffer[0] := DLE;
        ByteBuffer[1] := ACK;
        ByteBuffer[2] := STX;
        WriteData(ByteBuffer, 3);
        ReadData(ByteBuffer, 2);
        if (ByteBuffer[0] = DLE) and (ByteBuffer[1] = ACK) then begin
          ByteBuffer[0] := NUL;
          Move(Buffer^, ByteBuffer[1], Anzahl);
          for bDummy := 0 to Anzahl do begin
            WriteData(ByteBuffer[bDummy], 1);
            if ByteBuffer[bDummy] = DLE then
              WriteData(ByteBuffer[bDummy], 1);
          end;
          ByteBuffer[0] := DLE;
          ByteBuffer[1] := EOT;
          WriteData(ByteBuffer, 2);
          ReadData(ByteBuffer, 3);
          if (ByteBuffer[0] = DLE) and (ByteBuffer[1] = ACK) and (ByteBuffer[2] = STX) then begin
            ByteBuffer[0] := DLE;
            ByteBuffer[1] := ACK;
            WriteData(ByteBuffer, 2);
            ReadData(ByteBuffer, 3);
            if (ByteBuffer[0] = DC2) and (ByteBuffer[1] = DLE) and (ByteBuffer[2] = ETX) then begin
              ByteBuffer[0] := DLE;
              ByteBuffer[1] := ACK;
              WriteData(ByteBuffer, 2);
              Result := True;
            end;
          end;
        end;
      end;
    end;
  end;
end;

{------------------------------------------------------------------------------}

function  TS5Link.AgBausteinDir(var PLC: TPLC; BstTyp: Byte; Buffer: Pointer) : Boolean;
var wDummy : Word;
begin
  Result := False;
  if FunktionEinleiten(PLC, $1B) then begin
    ByteBuffer[0] := BausteinType[BstTyp];
    wDummy := 1;
    if BstTyp = OB then begin
      ByteBuffer[wDummy] := DLE;
      Inc(wDummy);
    end;
    ByteBuffer[wDummy] := DLE;
    Inc(wDummy);
    ByteBuffer[wDummy] := EOT;
    Inc(wDummy);
    WriteData(ByteBuffer, wDummy);
    if FunktionFortfuehren(PLC) then begin
      for wDummy := 0 to 511 do begin
        ReadData(ByteBuffer[wDummy], 1);
        if ByteBuffer[wDummy] = DLE then
          ReadData(ByteBuffer[wDummy], 1);
      end;
      Move(ByteBuffer, Buffer^, 512);
      if FunktionAbschliessen(PLC) then
         Result := True;
    end;
  end;
end;

{------------------------------------------------------------------------------}

function  TS5Link.AgReadTimer(var PLC : TPLC; TimNr : Byte; Buffer : Pointer) : Boolean;
var WordBuffer : Array[0..2] of Word;
begin
  Result := AgRead(PLC, PLC.System.AdrT + (TimNr SHL 1), PLC.System.AdrT + (TimNr SHL 1) + 1, @WordBuffer);
  if Result then
    Move(WordBuffer, Buffer^, 2);
end;

{------------------------------------------------------------------------------}

function  TS5Link.AgReadCounter(var PLC : TPLC; CntNr : Byte; Buffer : Pointer) : Boolean;
var WordBuffer : Word;
begin
  Result := AgRead(PLC, PLC.System.AdrZ + (CntNr SHL 1), PLC.System.AdrZ + (CntNr SHL 1) + 1, @WordBuffer);
  if Result then
    Move(WordBuffer, Buffer^, 2);
end;

{------------------------------------------------------------------------------}

function  TS5Link.AgReadEB(var PLC: TPLC; ByteNr: Byte; Buffer: Pointer) : Boolean;
begin
  Result := AGReadData(PLC, E, ByteNr, ByteNr, 1, Buffer);
end;

{------------------------------------------------------------------------------}

function  TS5Link.AgReadEW(var PLC: TPLC; WortNr: Byte; Buffer: Pointer) : Boolean;
var wDummy : Word;
begin
  Result := AGReadData(PLC, E, WortNr, WortNr, 2, @wDummy);
  // wDummy := Swap(wDummy);  Ansichtssache ...
  Move(wDummy, Buffer^, 2);
end;

{------------------------------------------------------------------------------}

function  TS5Link.AgReadAB(var PLC: TPLC; ByteNr: Byte; Buffer: Pointer) : Boolean;
begin
  Result := AGReadData(PLC, A, ByteNr, ByteNr, 1, Buffer);
end;

{------------------------------------------------------------------------------}

function  TS5Link.AgReadAW(var PLC: TPLC; WortNr: Byte; Buffer: Pointer) : Boolean;
var wDummy : Word;
begin
  Result := AGReadData(PLC, A, WortNr, WortNr, 2, @wDummy);
  // wDummy := Swap(wDummy); Ansichtssache ...
  Move(wDummy, Buffer^, 2);
end;

{------------------------------------------------------------------------------}

function  TS5Link.AgReadMB(var PLC: TPLC; ByteNr: Byte; Buffer: Pointer) : Boolean;
begin
  Result := AGReadData(PLC, M, ByteNr, ByteNr, 1, Buffer);
end;

{------------------------------------------------------------------------------}

function  TS5Link.AgReadMW(var PLC: TPLC; WortNr: Byte; Buffer: Pointer) : Boolean;
var wDummy : Word;
begin
  Result := AGReadData(PLC, M, WortNr, WortNr, 2, @wDummy);
  wDummy := Swap(wDummy);
  Move(wDummy, Buffer^, 2);
end;

{------------------------------------------------------------------------------}

function  TS5Link.AgWriteAB(var PLC: TPLC; ByteNr: Byte; Buffer: Pointer) : Boolean;
begin
  Result := AGWriteData(PLC, A, ByteNr, ByteNr, 1, Buffer);
end;

{------------------------------------------------------------------------------}

function  TS5Link.AgWriteAW(var PLC: TPLC; WortNr: Byte; Buffer: Pointer) : Boolean;
var wDummy : Word;
begin
  Move(Buffer^, wDummy, 2);
  wDummy := Swap(wDummy);
  Result := AGWriteData(PLC, A, WortNr, WortNr, 2, @wDummy);
end;

{------------------------------------------------------------------------------}

function  TS5Link.AgWriteMB(var PLC: TPLC; ByteNr: Byte; Buffer: Pointer) : Boolean;
begin
  Result := AGWriteData(PLC, M, ByteNr, ByteNr, 1, Buffer);
end;

{------------------------------------------------------------------------------}

function  TS5Link.AgWriteMW(var PLC: TPLC; WortNr: Byte; Buffer: Pointer) : Boolean;
var wDummy : Word;
begin
  Move(Buffer^, wDummy, 2);
  wDummy := Swap(wDummy);
  Result := AGWriteData(PLC, M, WortNr, WortNr, 2, @wDummy);
end;

{------------------------------------------------------------------------------}

function TS5Link.AgReadDataBlock(var PLC : TPLC; BstNr : Byte; AbWort, Anzahl : Word; Buffer : Pointer) : Boolean;
var BstInfo    : TBstInfo;
    wAnfAdr    : Word;
    wEndAdr    : Word;
    bDummy     : Byte;
begin
  Result := False;
  if AgBausteinInfo(PLC, DB, BstNr, @BstInfo) then begin
    wAnfAdr := BstInfo.BausteinAdresse + (Abwort SHL 1);
    if Anzahl > ((Hi(BstInfo.LaengeMitKopf) - (PLC.System.BstKopfLaenge SHR 1))) then
     Anzahl := (Hi(BstInfo.LaengeMitKopf) - (PLC.System.BstKopfLaenge SHR 1));
    wEndAdr := WanfAdr + (Anzahl  SHL 1) - 1;
    Result := AgRead(PLC, wAnfAdr, wEndAdr, @awBuffer);
    for bDummy := 0 to Anzahl do
      awBuffer[bDummy] := Swap(awBuffer[bDummy]);
    Move(awBuffer, Buffer^, Anzahl SHL 1);
  end;
end;

{------------------------------------------------------------------------------}

function  TS5Link.AgWriteDataBlock(var PLC : TPLC; BstNr : Byte; AbWort, Anzahl : Word; Buffer : Pointer) : Boolean;
var BstInfo   : TBstInfo;
    wAnfAdr   : Word;
    wEndAdr   : Word;
    bDummy     : Byte;
begin
  Result := False;
  Move(Buffer^, awBuffer, Anzahl  SHL 1);
  for bDummy := 0 to (Anzahl - 1) SHL 1 do
    awBuffer[bDummy] := Swap(awBuffer[bDummy]);
  if AgBausteinInfo(PLC, DB, BstNr, @BstInfo) then begin
    wAnfAdr := BstInfo.BausteinAdresse + (Abwort SHL 1);
    wEndAdr := WanfAdr + Anzahl SHL 1;
    Result := AgWrite(PLC, wAnfAdr, wEndAdr, @awBuffer)
  end;
end;

{------------------------------------------------------------------------------}

function TS5Link.AgStart(var PLC: TPLC) : Boolean;
var bStatus : Byte;
begin
  AgRead(PLC, PLC.System.AdrSD + $0C, PLC.System.AdrSD + $0C, @bStatus);
  bStatus := SetBit(bStatus, 5);
  bStatus := ClearBit(bStatus , 7);
  Result := AgWrite(PLC, PLC.System.AdrSD + $0C, PLC.System.AdrSD + $0C, @bStatus);
end;

{------------------------------------------------------------------------------}

function TS5Link.AgStop(var PLC: TPLC) : Boolean;
var bStatus : Byte;
begin
  AgRead(PLC, PLC.System.AdrSD + $0C, PLC.System.AdrSD + $0C, @bStatus);
  bStatus := SetBit(bStatus , 7);
  Result := AgWrite(PLC, PLC.System.AdrSD + $0C, PLC.System.AdrSD + $0C, @bStatus);
end;

{------------------------------------------------------------------------------}

function TS5Link.AgCompress(var PLC: TPLC) : Boolean;
begin
  Result := False;
  if FunktionEinleiten(PLC, $7) then begin
    ByteBuffer[0] := DLE;
    ByteBuffer[1] := EOT;
    WriteData(ByteBuffer, 2);
    ReadData(ByteBuffer, 2);
    if (ByteBuffer[0] = DLE) and (ByteBuffer[1] = ACK) then
      if FunktionAbschliessen(PLC) then
        Result := True;
  end;
end;

{------------------------------------------------------------------------------}

function  TS5Link.AgDelBst(var PLC: TPLC; BstTyp, BstNr : Byte) : Boolean;
var  bDummy : Byte;
begin
  Result := False;
  if FunktionEinleiten(PLC, $09) then begin
    bDummy := 0;
    ByteBuffer[bDummy] := BstTyp;
    Inc(bDummy);
    if ByteBuffer[bDummy] = $10 then begin
      ByteBuffer[bDummy] := DLE;
      Inc(bDummy);
    end;
    ByteBuffer[bDummy] := BstNr;
    Inc(bDummy);
    if ByteBuffer[bDummy] = $10 then begin
      ByteBuffer[bDummy] := DLE;
      Inc(bDummy);
    end;
    ByteBuffer[bDummy] := DLE;
    Inc(bDummy);
    ByteBuffer[bDummy] := ETX;
    Inc(bDummy);
    WriteData(ByteBuffer, bDummy);
    ReadData(ByteBuffer, 2);
    if (ByteBuffer[0] = DLE) and (ByteBuffer[1] = ACK) then
      if AbschlussMeldung(PLC) then
        Result := True;
  end;
end;

{------------------------------------------------------------------------------}

function TS5Link.CheckPLC(var PLC : TPLC) : Boolean;
var bStatus  : Byte;
begin
  Result := False;
  if AgRead(PLC, PLC.System.AdrSD + $0C, PLC.System.AdrSD + $0C, @bStatus) then
    Result := not TestBit(bStatus, 6);
end;

{------------------------------------------------------------------------------}

function TS5Link.CheckBatt(var PLC : TPLC) : Boolean;
var bStatus  : Byte;
begin
  Result := False;
  if AgRead(PLC, PLC.System.AdrSD + $0C, PLC.System.AdrSD + $0C, @bStatus) then
    Result := TestBit(bStatus, 3);
end;

{------------------------------------------------------------------------------}

function TS5Link.AgCreateBst(var PLC : TPLC; BstTyp, BstNr : Byte; Anzahl : Word) : Boolean;
begin
  with Baustein do begin
    SynchronMuster := $7070;                        // Synchronisationsmuster
    Kennung := BstTyp;                              // Bausteintyp
    Kennung :=  (Kennung SHL 8) and BstNr;          // Bausteinnummer
    PGKennung := $C0;                               // Sprachraum B (z.B AGS5-135U) + Baustein ohne Bib.-Nr.
    BibNr[1] := 0;                                  // Bibliotheksnummer 1.Byte
    BibNr[2] := 0;                                  // Bibliotheksnummer 2.Byte
    BibNr[3] := 0;                                  // Bibliotheksnummer 3.Byte
    LaengeMitKopf := Swap((PLC.System.BstKopfLaenge SHR 1) + Anzahl); // Lnge der Nutzdaten + Bausteinkopflnge in Worten
    FillChar(Daten, Anzahl SHL 1, #0);              // Nutzdatenbereich des Bausteins lschen ...
    Result := AgBausteinEingabe(PLC, BstTyp, BstNr, Swap(Baustein.LaengeMitKopf SHL 1), @Baustein);
  end;
end;

{------------------------------------------------------------------------------}

function TS5Link.GetErrText(wErrNum : Word) : String;
begin
  case wErrNum of
    S5_E_NOERR           : Result := csNoError;
    S5_E_WRONGMODE_A     : Result := csWrongMode;
    S5_E_UNKNOWNCMD      : Result := csUnknownCmd;
    S5_E_NOMEM           : Result := csNoMem;
    S5_E_BSTEXIST        : Result := csBstExist;
    S5_E_BSTEXISTINEPROM : Result := csBstExistInEPROM;
    S5_E_BSTLISTLOCKED   : Result := csBstListLocked;
    S5_E_NOBREAKPOINT    : Result := csNoBreakPoint;
    S5_E_WRONGMODE_B     : Result := csWrongMode;
    S5_E_BSTNOTEXIST     : Result := csBstNotExist;
    S5_E_NOINIT          : Result := csNoInit;
    S5_E_NOPORT          : Result := csNoPort;
    else Result                   := csDontKnow;
  end;
end;

{------------------------------------------------------------------------------}
function  TS5Link.GetPLCTyp(var PLC : TPLC) : String;
begin
  if PLC.Initialisiert then begin
    if (PLC.System.CPUKennung = $FF01) and (PLC.System.ZweiteCPUKennung = $0000) then
      Result := 'S5-90U         '
(*
    else if (PLC.System.CPUKennung = $FF01) and (PLC.System.ZweiteCPUKennung = $0000) then
      Result := 'S5-95U'
*)
(*
    else if (PLC.System.CPUKennung = $CF01) and (PLC.System.ZweiteCPUKennung = $0000) then
      Result := 'S5-100U  CPU 103'
    else if (PLC.System.CPUKennung = $CF01) and (PLC.System.ZweiteCPUKennung = $0000) then
      Result := 'S5-100U  CPU 103'
*)
    else if (PLC.System.CPUKennung = $CF01) and (PLC.System.ZweiteCPUKennung = $0000) then
      Result := 'S5-100U  CPU 103'
    else if (PLC.System.CPUKennung = $EF04) and (PLC.System.ZweiteCPUKennung = $0000) then
      Result := 'S5-115U  CPU 942'
(*
    else if (PLC.System.CPUKennung = $EF04) and (PLC.System.ZweiteCPUKennung = $0000) then
      Result := 'S5-115U  CPU 942'
    else if (PLC.System.CPUKennung = $EF04) and (PLC.System.ZweiteCPUKennung = $0000) then
      Result := 'S5-115U  CPU 943'
    else if (PLC.System.CPUKennung = $EF04) and (PLC.System.ZweiteCPUKennung = $0000) then
      Result := 'S5-115U  CPU 944'
*)
    else if (PLC.System.CPUKennung = $0240) and (PLC.System.ZweiteCPUKennung = $0127) then
      Result := 'S5-135U  CPU 928'
(*
    else if (PLC.System.CPUKennung = $0240) and (PLC.System.ZweiteCPUKennung = $0127) then
      Result := 'S5-135U  CPU 928'
    else if (PLC.System.CPUKennung = $0240) and (PLC.System.ZweiteCPUKennung = $0127) then
      Result := 'S5-135U  CPU 928'
    else if (PLC.System.CPUKennung = $0240) and (PLC.System.ZweiteCPUKennung = $0127) then
      Result := 'S5-155U  CPU 928'
    else if (PLC.System.CPUKennung = $0240) and (PLC.System.ZweiteCPUKennung = $0127) then
      Result := 'S5-155U  CPU 928'
    else if (PLC.System.CPUKennung = $0240) and (PLC.System.ZweiteCPUKennung = $0127) then
      Result := 'S5-155U  CPU 928'
*)
    else
      Result := 'Unbekannter Typ ';
  end else
    Result := '                ';
end;

{------------------------------------------------------------------------------}

function  TS5Link.AgDecodeBst(var PLC : TPLC; Bst : TBaustein; boAdressen : Boolean) : TStringList;
var wDummy     : Word;
    bDummy     : Byte;
    Zeile      : String;
    Rand       : String;
    wNetzwerk  : Word;
    Zeilen     : TStringList;
    Anzahl     : Word;
    awData     : Array[0..4096] of Word;
    boSonder   : Boolean;
    wOffset    : Word;
    abKopf     : Array[0..4096] of Byte;
    ParamCnt   : Byte;
    bOffset    : Byte;
    asParam    : Array[1..255] of String[4];

  {------------------------------------------------------------------------------}

  procedure DecodeEA(var Zeile : String; bIn : Byte);
  begin
    Zeile := Zeile + '    BI/BY/W/D : ';
    case bIn of
      $10 : Zeile := Zeile + 'D';
      $20 : Zeile := Zeile + 'W';
      $40 : Zeile := Zeile + 'BY';
      $80 : Zeile := Zeile + 'BI';
    end;
  end;

  {------------------------------------------------------------------------------}

  procedure DecodeD(var Zeile : String; bIn : Byte);
  begin
    Zeile := Zeile + '    BI/BY/W/D : ';
    case bIn of
      $01 : Zeile := Zeile + 'KZ';
      $02 : Zeile := Zeile + 'KT';
      $04 : Zeile := Zeile + 'KF';
      $08 : Zeile := Zeile + 'KG';
      $10 : Zeile := Zeile + 'KC';
      $20 : Zeile := Zeile + 'KY';
      $40 : Zeile := Zeile + 'KH';
      $80 : Zeile := Zeile + 'KM';
    end;
  end;

  {------------------------------------------------------------------------------}

  function DecodeZeile (wDummy : Word) : String;
  var bDummy     : Byte;
      Zeile      : String;
      Rand       : String;
      wOffset    : Word;
      ParamCnt   : Byte;
      bOffset    : Byte;
      Baustein   : TBaustein;

  begin
    Rand := '';
    case (Hi(awData[wDummy])) of
      $00 : Zeile := 'NOP 0';
      $01 : Zeile := 'KEW';
      $02 : Zeile := 'L T ' + IntToStr(Lo(awData[wDummy]));
      $03 : Zeile := 'TNB ' + IntToStr(Lo(awData[wDummy]));
      $04 : Zeile := 'FR T ' + IntToStr(Lo(awData[wDummy]));
      $05 : Zeile := 'BEB';
      $06 : Zeile := 'FR = ' +asParam[Lo(awData[wDummy])];
      $07 : Zeile := 'U = ' + asParam[Lo(awData[wDummy])];
      $08 : if Lo(awData[wDummy]) = 0 then
              Zeile := 'AS'
            else
              Zeile := 'AF';
      $09 : Zeile := 'KZW';
      $0A : Zeile := 'L MB ' + IntToStr(Lo(awData[wDummy]));
      $0B : Zeile := 'T MB ' + IntToStr(Lo(awData[wDummy]));
      $0C : Zeile := 'LC T ' + IntToStr(Lo(awData[wDummy]));
      $0D : Zeile := 'SPO = ' + IntToStr(Lo(awData[wDummy]));
      $0E : Zeile := 'LC = ' + asParam[Lo(awData[wDummy])];
      $0F : Zeile := 'O = ' + asParam[Lo(awData[wDummy])];
      $10 : begin
              if Lo(awData[wDummy]) = $82 then
                Zeile := ''         // BLD 130
              else if Lo(awData[wDummy]) = $83 then
                Zeile := 'AWL'
              else if Lo(awData[wDummy]) = $84 then
                Zeile := 'BLD 132'
              else if Lo(awData[wDummy]) = $85 then
                Zeile := 'BLD 133'
              else begin
                Zeile := '***';    /// BLD 255
                Inc(wNetzwerk);
              end;
            end;
      $11 : Zeile := 'I ' + IntToStr(Lo(awData[wDummy]));
      $12 : Zeile := 'L MW  ' + IntToStr(Lo(awData[wDummy]));
      $13 : Zeile := 'T MW  ' + IntToStr(Lo(awData[wDummy]));
      $14 : Zeile := 'SA T ' + IntToStr(Lo(awData[wDummy]));
      $15 : Zeile := 'SPP = ' + IntToStr(Lo(awData[wDummy]));
      $16 : Zeile := 'SAR = ' + IntToStr(Lo(awData[wDummy]));
      $17 : Zeile := 'S = ' + asParam[Lo(awData[wDummy])];
      $18 : Zeile := 'XXXX';
      $19 : Zeile := 'D ' + IntToStr(Lo(awData[wDummy]));
      $1A : Zeile := 'XXXX';
      $1B : Zeile := 'XXXX';
      $1C : Zeile := 'SV T ' + IntToStr(Lo(awData[wDummy]));
      $1D : Zeile := 'SPB FB ' + IntToStr(Lo(awData[wDummy]));
      $1E : Zeile := 'SVZ = ' + asParam[Lo(awData[wDummy])];
      $1F : Zeile := '== ' + asParam[Lo(awData[wDummy])];
      $20 : Zeile := 'A DB ' + IntToStr(Lo(awData[wDummy]));
      $21 : begin
              if Lo(awData[wDummy]) = $20 then
                Zeile := '>F'
              else if Lo(awData[wDummy]) = $40 then
                Zeile := '<F'
              else if Lo(awData[wDummy]) = $60 then
                Zeile := '><F'
              else if Lo(awData[wDummy]) = $80 then
                Zeile := '!=F'
              else if Lo(awData[wDummy]) = $A0 then
                Zeile := '>=F'
              else
                Zeile := '<=F';
            end;
      $22 : Zeile := 'L DL ' + IntToStr(Lo(awData[wDummy]));
      $23 : Zeile := 'T DL ' + IntToStr(Lo(awData[wDummy]));
      $24 : Zeile := 'SE T ' + IntToStr(Lo(awData[wDummy]));
      $25 : Zeile := 'SPM = ' + IntToStr(Lo(awData[wDummy]));
      $26 : Zeile := 'SE = ' + asParam[Lo(awData[wDummy])];
      $27 : Zeile := 'UN = ' + asParam[Lo(awData[wDummy])];
      $28 : Zeile := 'L KB ' + IntToStr(Lo(awData[wDummy]));
      $29 : Zeile := 'XXXX';
      $2A : Zeile := 'L DR ' + IntToStr(Lo(awData[wDummy]));
      $2B : Zeile := 'T DR ' + IntToStr(Lo(awData[wDummy]));
      $2C : Zeile := 'SS T ' + IntToStr(Lo(awData[wDummy]));
      $2D : begin
              if (Hi(awData[wDummy - 1]) = $1D) or (Hi(awData[wDummy - 1]) = $3D) then begin
                AgBausteinAusgabe(PLC, FB, Lo(awData[wDummy - 1]) , @Baustein);
                Rand :=  'Name : ';
                Zeile := '';
                wOffset := 2;
                bDummy := 0 + wOffset;
                repeat
                  Zeile := Zeile + Chr(Baustein.Daten[bDummy + 1]);
                  Zeile := Zeile + Chr(Baustein.Daten[bDummy]);
                  Inc(bDummy, 2);
                until bDummy = 8 + wOffset;
                if boAdressen then
                  Zeile := '      ' + Rand + Zeile;
                Zeilen.Add(Zeile);
                ParamCnt := (((Lo(Baustein.Daten[0]) SHL 1) - 8) DIV 6);
                if ParamCnt > 0 then begin
                  for wOffset := 0 to ParamCnt - 1 do begin
                    Rand := '';
                    Zeile := '';
                    bOffset := 12 + (wOffset * 6);
                    for bDummy := 0 to 1 do begin
                      Rand := Rand + Chr(Baustein.Daten[bOffset + 1]);
                      Rand := Rand + Chr(Baustein.Daten[bOffset]);
                      Inc(bOffset, 2);
                    end;
                    Zeile := DecodeZeile(wDummy + 1 + wOffset);
                    Zeile := Copy(Zeile, 3, Length(Zeile) - 2);
                    if boAdressen then
                      Zeile := '      ' + Rand + ' : ' + Zeile;
                    Zeilen.Add(Zeile);
                  end;
                end;
                Zeile := 'XXXX';
                for bDummy := 1 to 3 do
                  awData[wDummy + bDummy] := $FFFE;
              end else
                Zeile := 'SPA = ' + IntToStr(Lo(awData[wDummy]));
            end;
      $2E : Zeile := 'SSV = ' + asParam[Lo(awData[wDummy])];
      $2F : Zeile := 'ON = ' + asParam[Lo(awData[wDummy])];
      $30 : begin
              if Lo(awData[wDummy]) = $01 then begin
                Zeile := 'L KZ ' + IntToStr(awData[wDummy + 1]);
                awData[wDummy + 1] := $FFFE;
              end else if Lo(awData[wDummy]) = $02 then begin
                Zeile := 'L KT ' + IntToHex(awData[wDummy + 1] and $0FFF, 3) + '.' + IntToHex((awData[wDummy + 1] and $F000) SHR 12, 1);
                awData[wDummy + 1] := $FFFE;
              end else if Lo(awData[wDummy]) = $04 then begin
                Zeile := 'L KF ';
                if TestBit(awData[wDummy + 1], 15) then begin
                  Zeile := Zeile + '-';
                  Zeile := Zeile + IntToStr(65536 - awData[wDummy + 1]);
                end else begin
                  Zeile := Zeile + '+';
                  Zeile := Zeile + IntToStr(awData[wDummy + 1]);
              end;
                awData[wDummy + 1] := $FFFE;
              end else if Lo(awData[wDummy]) = $10 then begin
                Zeile := 'L KC ' + IntToHex(WordToBCD(awData[wDummy + 1]), 4);
                awData[wDummy + 1] := $FFFE;
              end else if Lo(awData[wDummy]) = $20 then begin
                Zeile := 'L KY ' + IntToStr(Hi(awData[wDummy])) + ',' + IntToStr(Lo(awData[wDummy]));
                awData[wDummy + 1] := $FFFE;
              end else if Lo(awData[wDummy]) = $40 then begin
                Zeile := 'L KH ' + IntToHex(awData[wDummy + 1], 4);
                awData[wDummy + 1] := $FFFE;
              end else begin
                Zeile := 'L KM ' + WordToBin(awData[wDummy + 1]);
                awData[wDummy + 1] := $FFFE;
              end;
            end;
      $31 : Zeile := 'XXXX';
      $32 : Zeile := 'L DW ' + IntToStr(Lo(awData[wDummy]));
      $33 : Zeile := 'T DW ' + IntToStr(Lo(awData[wDummy]));
      $34 : Zeile := 'SI T ' + IntToStr(Lo(awData[wDummy]));
      $35 : Zeile := 'SPN = ' + IntToStr(Lo(awData[wDummy]));
      $36 : Zeile := 'SI = ' + asParam[Lo(awData[wDummy])];
      $37 : Zeile := 'RB = ' + asParam[Lo(awData[wDummy])];
      $38 : Zeile := 'XXXX';
      $39 : Zeile := 'XXXX';
      $3A : Zeile := 'XXXX';
      $3B : Zeile := 'XXXX';
      $3C : Zeile := 'R T ' + IntToStr(Lo(awData[wDummy]));
      $3D : Zeile := 'SPA FB ' + IntToStr(Lo(awData[wDummy]));
      $3E : Zeile := 'RD = ' + asParam[Lo(awData[wDummy])];
      $3F : Zeile := 'LW = ' + asParam[Lo(awData[wDummy])];
      $40 : Zeile := 'LIR ' + IntToStr(Lo(awData[wDummy]));
      $41 : Zeile := 'UW';
      $42 : Zeile := 'L Z ' + IntToStr(Lo(awData[wDummy]));
      $43 : Zeile := 'XXXX';
      $44 : Zeile := 'FR Z ' + IntToStr(Lo(awData[wDummy]));
      $45 : Zeile := 'SPZ = ' + IntToStr(Lo(awData[wDummy]));
      $46 : Zeile := 'L = ' + asParam[Lo(awData[wDummy])];
      $47 : Zeile := 'XXXX';
      $48 : Zeile := 'TIR ' + IntToStr(Lo(awData[wDummy]));
      $49 : Zeile := 'OW';
      $4A : if (Lo(awData[wDummy]) and $80 = $80) then
              Zeile := 'L AB ' + IntToStr(Lo(awData[wDummy]) and not $80)
            else
              Zeile := 'L EB ' + IntToStr(Lo(awData[wDummy]));
      $4B : if (Lo(awData[wDummy]) and $80 = $80) then
              Zeile := 'T AB ' + IntToStr(Lo(awData[wDummy]) and not $80)
            else
              Zeile := 'T EB ' + IntToStr(Lo(awData[wDummy]));
      $4C : Zeile := 'LC Z' + IntToStr(Lo(awData[wDummy]));
      $4D : Zeile := 'SPB OB ' + IntToStr(Lo(awData[wDummy]));
      $4E : Zeile := 'B MW ' + IntToStr(Lo(awData[wDummy]));
      $4F : Zeile := 'XXXX';
      $50 : Zeile := 'ADD BF ' + IntToStr(Lo(awData[wDummy]));
      $51 : Zeile := 'XOW ';
      $52 : if (Lo(awData[wDummy]) and $80 = $80) then
              Zeile := 'L AW ' + IntToStr(Lo(awData[wDummy]) and not $80)
            else
              Zeile := 'L EW ' + IntToStr(Lo(awData[wDummy]));
      $53 : if (Lo(awData[wDummy]) and $80 = $80) then
              Zeile := 'T AW ' + IntToStr(Lo(awData[wDummy]) and not $80)
            else
              Zeile := 'T EW ' + IntToStr(Lo(awData[wDummy]));
      $54 : Zeile := 'ZR Z ' + IntToStr(Lo(awData[wDummy]));
      $55 : Zeile := 'SPB PB ' + IntToStr(Lo(awData[wDummy]));
      $56 : Zeile := 'XXXX';
      $57 : Zeile := 'XXXX';
      $58 : begin
              Zeile := 'ADD KF ';
                if TestBit(awData[wDummy + 1], 15) then
                  Zeile := Zeile + '-'
                else
                  Zeile := Zeile + '+';
                ClearBit(awData[wDummy + 1], 15);
                Zeile := Zeile + IntToStr(awData[wDummy + 1]);
                awData[wDummy + 1] := $FFFE;
            end;
      $59 : Zeile := '-F';
      $5A : Zeile := 'XXXX';
      $5B : Zeile := 'XXXX';
      $5C : Zeile := 'S Z ' + IntToStr(Lo(awData[wDummy]));
      $5D : Zeile := 'SPB SB ' + IntToStr(Lo(awData[wDummy]));
      $5E : Zeile := 'XXXX';
      $5F : Zeile := 'XXXX';
      $60 : Zeile := 'XXXX';
      $61 : Zeile := 'SLW ' + IntToStr(Lo(awData[wDummy]));
      $62 : Zeile := 'L BS ' + IntToStr(Lo(awData[wDummy]));
      $63 : Zeile := 'T BS ' + IntToStr(Lo(awData[wDummy]));
      $64 : Zeile := 'XXXX';
      $65 : if Lo(awData[wDummy]) = 0 then
              Zeile := 'BE'
            else
              Zeile := 'BEA';
      $66 : Zeile := 'T = ' + asParam[Lo(awData[wDummy])];
      $67 : Zeile := 'XXXX';
      $68 : Zeile := 'XXXX';
      $69 : Zeile := 'SRW ' + IntToStr(Lo(awData[wDummy]));
      $6A : Zeile := 'XXXX';
      $6B : Zeile := 'XXXX';
      $6C : Zeile := 'ZV Z ' + IntToStr(Lo(awData[wDummy]));
      $6D : Zeile := 'SPA OB ' + IntToStr(Lo(awData[wDummy]));
      $6E : Zeile := 'B DW ' + IntToStr(Lo(awData[wDummy]));
      $6F : Zeile := 'XXXX';
      $70 : begin
              if Lo(awData[wDummy]) = $00 then
                Zeile := 'STS'
              else if Lo(awData[wDummy]) = $02 then
                Zeile := 'TAK'
              else if Lo(awData[wDummy]) = $03 then
                Zeile := 'STP'
              else if Lo(awData[wDummy]) = $0B then begin
                Zeile := 'SPR ' +  IntToStr(awData[wDummy + 1]);
                awData[wDummy + 1] := $FFFE;
              end else if Lo(awData[wDummy]) = $15 then begin
                if Hi(awData[wDummy + 1]) = $00 then
                  Zeile := 'RU Z ' + IntToSTr(Lo(awData[wDummy + 1]))
                else if Hi(awData[wDummy + 1]) = $40 then
                  Zeile := 'SU Z ' + IntToSTr(Lo(awData[wDummy + 1]))
                else if Hi(awData[wDummy + 1]) = $80 then
                  Zeile := 'PN Z ' + IntToSTr(Lo(awData[wDummy + 1]))
                else if Hi(awData[wDummy + 1]) = $C0 then
                  Zeile := 'P Z ' + IntToSTr(Lo(awData[wDummy + 1]));
              end else if Lo(awData[wDummy]) = $25 then begin
                if Hi(awData[wDummy + 1]) = $00 then
                  Zeile := 'RU T ' + IntToSTr(Lo(awData[wDummy + 1]))
                else if Hi(awData[wDummy + 1]) = $40 then
                  Zeile := 'SU T ' + IntToSTr(Lo(awData[wDummy + 1]))
                else if Hi(awData[wDummy + 1]) = $80 then
                  Zeile := 'PN T ' + IntToSTr(Lo(awData[wDummy + 1]))
                else if Hi(awData[wDummy + 1]) = $C0 then
                  Zeile := 'P T ' + IntToSTr(Lo(awData[wDummy + 1]));
              end else if Lo(awData[wDummy]) = $46 then begin
                if Hi(awData[wDummy + 1]) = $00 then
                  Zeile := 'RU D ' + IntToSTr(Lo(awData[wDummy + 1]))
                else if Hi(awData[wDummy + 1]) = $40 then
                  Zeile := 'SU D ' + IntToSTr(Lo(awData[wDummy + 1]))
                else if Hi(awData[wDummy + 1]) = $80 then
                  Zeile := 'PN D ' + IntToSTr(Lo(awData[wDummy + 1]))
                else if Hi(awData[wDummy + 1]) = $C0 then
                  Zeile := 'P D ' + IntToSTr(Lo(awData[wDummy + 1]));
              end else if Lo(awData[wDummy]) = $57 then begin
                if Hi(awData[wDummy + 1]) = $00 then
                  Zeile := 'RU BS ' + IntToSTr(Lo(awData[wDummy + 1]))
                else if Hi(awData[wDummy + 1]) = $40 then
                  Zeile := 'SU BS ' + IntToSTr(Lo(awData[wDummy + 1]))
                else if Hi(awData[wDummy + 1]) = $80 then
                  Zeile := 'PN BS ' + IntToSTr(Lo(awData[wDummy + 1]))
                else if Hi(awData[wDummy + 1]) = $C0 then
                  Zeile := 'P BS ' + IntToSTr(Lo(awData[wDummy + 1]));
              end;
              awData[wDummy + 1] := $FFFE;
            end;
      $71 : Zeile := 'XXXX';
      $72 : Zeile := 'L PB ' + IntToStr(Lo(awData[wDummy]));
      $73 : Zeile := 'T PB ' + IntToStr(Lo(awData[wDummy]));
      $74 : Zeile := 'XXXX';
      $75 : Zeile := 'SPA PB ' + IntToStr(Lo(awData[wDummy]));
      $76 : Zeile := 'B = ' + asParam[Lo(awData[wDummy])];
      $77 : Zeile := 'XXXX';
      $78 : if Lo(awData[wDummy]) = $05 then begin
              Zeile := 'E DB' + IntToStr(Lo(awData[wDummy + 1]));
              awData[wDummy + 1] := $FFFE;
            end else
              Zeile := 'XXXX';
      $79 : Zeile := '+F';
      $7A : Zeile := 'L PW ' + IntToStr(Lo(awData[wDummy]));
      $7B : Zeile := 'T PW ' + IntToStr(Lo(awData[wDummy]));
      $7C : Zeile := 'R Z ' + IntToStr(Lo(awData[wDummy]));
      $7D : Zeile := 'SPA SB ' + IntToStr(Lo(awData[wDummy]));
      $7E : Zeile := 'XXXX';
      $7F : Zeile := 'XXXX';
      $80..
      $87 : Zeile := 'U M ' + IntToStr(Lo(awData[wDummy])) + '.' + IntToStr(Hi(awData[wDummy]) and $07);
      $88..
      $8F : Zeile := 'O M ' + IntToStr(Lo(awData[wDummy])) + '.' + IntToStr(Hi(awData[wDummy]) and $07);
      $90..
      $97 : Zeile := 'S M ' + IntToStr(Lo(awData[wDummy])) + '.' + IntToStr(Hi(awData[wDummy]) and $07);
      $98..
      $9F : Zeile := '= M ' + IntToStr(Lo(awData[wDummy])) + '.' + IntToStr(Hi(awData[wDummy]) and $07);
      $A0..
      $A7 : Zeile := 'UN M ' + IntToStr(Lo(awData[wDummy])) + '.' + IntToStr(Hi(awData[wDummy]) and $07);
      $A8..
      $AF : Zeile := 'ON M ' + IntToStr(Lo(awData[wDummy])) + '.' + IntToStr(Hi(awData[wDummy]) and $07);
      $B0..
      $B7 : Zeile := 'R M ' + IntToStr(Lo(awData[wDummy])) + '.' + IntToStr(Hi(awData[wDummy]) and $07);
      $B8 : Zeile := 'U Z ' + IntToStr(Lo(awData[wDummy]));
      $B9 : Zeile := 'O Z ' + IntToStr(Lo(awData[wDummy]));
      $BA : Zeile := 'U (';
      $BB : Zeile := 'O (';
      $BC : Zeile := 'UN Z ' + IntToStr(Lo(awData[wDummy]));
      $BD : Zeile := 'ON Z ' + IntToStr(Lo(awData[wDummy]));
      $BE : Zeile := 'XXXX';
      $BF : Zeile := ')';
      $C0 ..
      $C7 : if (Lo(awData[wDummy]) and $80 = $80) then
              Zeile := 'U A ' + IntToStr(Lo(awData[wDummy]) and not $80) + '.' + IntToStr(Hi(awData[wDummy]) and $07)
            else
              Zeile := 'U E ' + IntToStr(Lo(awData[wDummy])) + '.' + IntToStr(Hi(awData[wDummy]) and $07);
      $C8..
      $CF : if (Lo(awData[wDummy]) and $80 = $80) then
              Zeile := 'O A ' + IntToStr(Lo(awData[wDummy]) and not $80) + '.' + IntToStr(Hi(awData[wDummy]) and $07)
            else
              Zeile := 'O E ' + IntToStr(Lo(awData[wDummy])) + '.' + IntToStr(Hi(awData[wDummy]) and $07);
      $D0..
      $D7 : if (Lo(awData[wDummy]) and $80 = $80) then
              Zeile := 'S A ' + IntToStr(Lo(awData[wDummy]) and not $80) + '.' + IntToStr(Hi(awData[wDummy]) and $07)
            else
              Zeile := 'S E ' + IntToStr(Lo(awData[wDummy])) + '.' + IntToStr(Hi(awData[wDummy]) and $07);
      $D8..
      $DF : if (Lo(awData[wDummy]) and $80 = $80) then
              Zeile := '= A ' + IntToStr(Lo(awData[wDummy]) and not $80) + '.' + IntToStr(Hi(awData[wDummy]) and $07)
            else
              Zeile := '= E ' + IntToStr(Lo(awData[wDummy])) + '.' + IntToStr(Hi(awData[wDummy]) and $07);
      $E0..
      $E7 : if (Lo(awData[wDummy]) and $80 = $80) then
              Zeile := 'UN A ' + IntToStr(Lo(awData[wDummy]) and not $80) + '.' + IntToStr(Hi(awData[wDummy]) and $07)
            else
              Zeile := 'UN E ' + IntToStr(Lo(awData[wDummy])) + '.' + IntToStr(Hi(awData[wDummy]) and $07);
      $E8..
      $EF : if (Lo(awData[wDummy]) and $80 = $80) then
              Zeile := 'ON A ' + IntToStr(Lo(awData[wDummy]) and not $80) + '.' + IntToStr(Hi(awData[wDummy]) and $07)
            else
              Zeile := 'ON E ' + IntToStr(Lo(awData[wDummy])) + '.' + IntToStr(Hi(awData[wDummy]) and $07);
      $F0..
      $F7 : if (Lo(awData[wDummy]) and $80 = $80) then
              Zeile := 'R A ' + IntToStr(Lo(awData[wDummy]) and not $80) + '.' + IntToStr(Hi(awData[wDummy]) and $07)
            else
              Zeile := 'R E ' + IntToStr(Lo(awData[wDummy])) + '.' + IntToStr(Hi(awData[wDummy]) and $07);
      $F8 : Zeile := 'U T ' + IntToStr(Lo(awData[wDummy]));
      $F9 : Zeile := 'O T ' + IntToStr(Lo(awData[wDummy]));
      $FA : Zeile := 'SPB = ' + IntToStr(Lo(awData[wDummy]));
      $FB : Zeile := 'O';
      $FC : Zeile := 'UN T ' + IntToStr(Lo(awData[wDummy]));
      $FD : Zeile := 'ON T ' + IntToStr(Lo(awData[wDummy]));
      $FE : Zeile := 'XXXX';
      $FF : if Lo(awData[wDummy]) = $FF then
              Zeile := 'NOP 1'
            else
              Zeile := 'XXXX';
    end;
    Result := Zeile;
  end;

  {------------------------------------------------------------------------------}

  begin
  Zeilen := TStringList.Create;
  try
    boSonder := False;
    case (Hi(Bst.Kennung) and $3f) of
      $01 : Zeile := 'DB ';
      $02 : Zeile := 'SB ';
      $04 : Zeile := 'PB ';
      $05 : begin
              Zeile := 'FX ';
              boSonder := True;
            end;
      $08 : begin
              Zeile := 'FB ';
              boSonder := True;
            end;
      $0C : Zeile := 'DX ';
      $10 : Zeile := 'OB ';
    end;
    Zeile := Zeile + IntToStr(Lo(Bst.Kennung));
    Zeilen.Add(Zeile);
    wNetzwerk := 1;
    Zeilen.Add('Netzwerk ' + IntToStr(wNetzwerk));
    Zeilen.Add('');
    Anzahl := Swap(Bst.LaengeMitKopf) - (PLC.System.BstKopflaenge SHR 1) - 1;
    Move(Bst.Daten, awData, (Anzahl + 1) SHL 1);
    wOffset := 0;
    if boSonder then begin
      wOffset := Lo(awData[0]);
      Zeile := 'Name : ';
      Move(awData[1], abKopf[0], wOffset SHL 1);
      bDummy := 0;
      repeat
       Zeile := Zeile + Chr(abKopf[bDummy + 1]);
       Zeile := Zeile + Chr(abKopf[bDummy]);
       Inc(bDummy, 2);
      until bDummy = 8;
      Zeilen.Add(Zeile);
      ParamCnt := (((Lo(awData[0]) SHL 1) - 8) DIV 6);
      if ParamCnt > 0 then begin
        for wDummy := 0 to ParamCnt - 1 do begin
          Zeile := 'Bez  : ';
          bOffset := 10 + (wDummy * 6);
          for bDummy := 0 to 1 do begin
           Zeile := Zeile + Chr(abKopf[bOffset + 1]);
           Zeile := Zeile + Chr(abKopf[bOffset]);
           Inc(bOffset, 2);
          end;
          asParam[wDummy + 1] := Copy(Zeile, Length(Zeile) - 4, 4);
          Zeile := Zeile + '    E/A/D/B/T/Z : ';
          case abKopf[9 + (wDummy * 6)] of
            $2 : begin
                   Zeile := Zeile + 'E  ';
                   DecodeEA(Zeile, abKopf[8 + (wDummy * 6)]);
                 end;
            $4 : begin
                   Zeile := Zeile + 'D  ';
                   DecodeD(Zeile, abKopf[8 + (wDummy * 6)]);
                 end;
            $5 : Zeile := Zeile + 'T  ';
            $6 : Zeile := Zeile + 'Z  ';
            $7 : Zeile := Zeile + 'B  ';
            $8 : begin
                   Zeile := Zeile + 'A  ';
                   DecodeEA(Zeile, abKopf[8 + (wDummy * 6)]);
                 end;
          end;
          Zeilen.Add(Zeile);
        end;
      end;
      Zeilen.Add('');
    end;
    for wDummy := wOffset to Anzahl do begin
      Zeile := DecodeZeile(wDummy);
      if Zeile <> 'XXXX' then begin
        if Rand = '' then
          Rand := '     : ';
        if boAdressen then
          Zeile := IntToHex(wDummy, 4) + '  ' + Rand + Zeile;
        Zeilen.Add(Zeile);
      end;
      if Copy(Zeile, Length(Zeile) - 2, 3) = '***' then begin
        Zeilen.Add('');
        Zeilen.Add('Netzwerk ' + IntToStr(wNetzwerk));
        Zeilen.Add('');
      end;
    end;
    Result := Zeilen;
  finally
    {Zeilen.Free;}
  end;
end;

{--- Hilfsfunktionen und Utils ------------------------------------------------}

function SureDlg(sText : String) : Boolean;
begin
  Result :=  (MessageDlg(sText, mtConfirmation, [mbYes, mbNo], 0) = mrYes);
end;

{------------------------------------------------------------------------------}

procedure InfoDlg(sText : String);
begin
  MessageDlg(sText, mtInformation, [mbOk], 0);
end;

{------------------------------------------------------------------------------}

procedure ShowInfo(sCaption, sInfo : String);
begin
  InfoForm := TInfoForm.Create(Application);
  try
    with InfoForm do begin
     Caption := sCaption;
     Label1.Caption := sInfo;
     Width := Label1.Left + Label1.Canvas.TextWidth(sInfo) + 20;
     Show;
     Update;
    end;
  finally
  end;
end;

{------------------------------------------------------------------------------}

procedure   HideInfo;
begin
  InfoForm.Close;
end;

{------------------------------------------------------------------------------}
procedure DecodeTimer(wTimer : Word; var TimerVal, TimerRes : Word);
begin
  TimerVal := Swap(wTimer) AND $07FF;
  TimerRes := (Swap(wTimer) AND $3000) SHR 12;
end;

{------------------------------------------------------------------------------}

procedure DecodeCounter(wCounter : Word; var CounterVal : Word);
begin
  CounterVal := Swap(wCounter) AND $07FF;
end;

{------------------------------------------------------------------------------}

function Exponent(Base, Expo: Integer) : LongInt;
var  iDummy : Integer;
begin
  Result := 1;
  if (Base < 0) or (Expo < 0) then
    Result := -1;
  if Result = 1 then
    for iDummy := 1 to Expo do begin
      Result := Base * Result;
    end;
end;

{------------------------------------------------------------------------------}

function HexCharToWert(HexChar: Char) : Word;
begin
  case HexChar of
    'A': Result := 10;
    'B': Result := 11;
    'C': Result := 12;
    'D': Result := 13;
    'E': Result := 14;
    'F': Result := 15;
    else Result := StrToInt(HexChar);
  end;
end;

{------------------------------------------------------------------------------}

function HexToInt(HexWert: String): LongInt;
var iDummy : Integer;
begin
  Result := 0;
  HexWert := UpperCase(HexWert);
  for iDummy:= 1 to Length(HexWert) do
    case Ord(HexWert[iDummy]) of
      0..47,
      58..64,
      71..255 : Result := -1
    end;
  if Result = 0 then
    for iDummy := 0 to length(HexWert) -1 do
      Result := Result + (HexCharToWert(HexWert[Length(HexWert) - iDummy]) *
                          Exponent(16, iDummy));
end;

{------------------------------------------------------------------------------}

function BinToInt(BinWert: String) : LongInt;
var  iDummy : Integer;
begin
  Result := 0;
  BinWert := UpperCase(BinWert);
  for iDummy := 1 to Length(BinWert) do
    case Ord(BinWert[iDummy]) of
      0..47,
      50..255 : Result := -1
    end;
  if Result = 0 then
    for iDummy:= 0 to Length(BinWert) -1 do
      Result := Result + (StrToInt(BinWert[Length(BinWert) - iDummy]) *
                          Exponent(2, iDummy));
end;

{------------------------------------------------------------------------------}

function IntToBin(IntWert: LongInt) : String;
var lDiv, lMod, lZahl : LongInt;
begin
  Result := '';
  if IntWert >= 0 then begin
    lZahl := IntWert;
    Repeat
      lDiv := lZahl div 2;
      lMod := lZahl mod 2;
      Result := IntToStr(lMod) + Result;
      lZahl := lDiv;
    Until lDiv = 0;
  end;
end;

{------------------------------------------------------------------------------}

function TestBit(wIn : Word; bBit : Byte) : Boolean;
begin
  Result := (wIn and (1 SHL bBit) = (1 SHL bBit));
end;

{------------------------------------------------------------------------------}

function SetBit(wIn : Word; bBit : Byte) : Word;
begin
  Result  := wIn or (1 SHL bBit);
end;

{------------------------------------------------------------------------------}

function ClearBit(wIn : Word; bBit : Byte) : Word;
begin
  Result := wIn and not (1 SHL bBit);
end;

{------------------------------------------------------------------------------}

function ChangeBit(wIn : Word; bBit : Byte) : Word;
begin
  Result := wIn xor (1 SHL bBit);
end;

{------------------------------------------------------------------------------}

function WordToBin(wData : Word) : String;
var  bDummy  : Byte;
begin
  Result := '0000000000000000';
  for bDummy := 0 to 15 do
    if TestBit(wData, bDummy) then
      Result[16 - bDummy] := '1';
end;

{------------------------------------------------------------------------------}

function ByteToBin(bData : Byte) : String;
var  bDummy  : Byte;
begin
  Result := '00000000';
  for bDummy := 0 to 7 do
    if TestBit(bData, bDummy) then
      Result[8 - bDummy] := '1';
end;

{------------------------------------------------------------------------------}

function ByteToBCD(bDez : Byte) : Byte;
begin
  Result := ((bDez DIV 10) SHL 4) OR bDez MOD 10;
end;

{------------------------------------------------------------------------------}

function WordToBCD(wDez : Word) : Word;
begin
  Result := (ByteToBCD(wDez DIV 100)) SHL 8 or (ByteToBCD((wDez MOD 100)));
end;

{------------------------------------------------------------------------------}

procedure Register;
begin
  RegisterComponents('SPS', [TS5Link]);
end;

{------------------------------------------------------------------------------}

end.
